package gov.va.med.mhv.calendar.service.impl;

import gov.va.med.mhv.calendar.dto.CalendarCategoryDTO;
import gov.va.med.mhv.calendar.dto.CalendarEventDTO;
import gov.va.med.mhv.calendar.dto.CalendarSearchCriteriaDTO;
import gov.va.med.mhv.calendar.dto.CategoryDTO;
import gov.va.med.mhv.calendar.dto.MHVEventModel;
import gov.va.med.mhv.calendar.dto.ToDoTaskDTO;
import gov.va.med.mhv.calendar.dto.UserCalViewDTO;
import gov.va.med.mhv.calendar.service.CalendarEventService;
import gov.va.med.mhv.calendar.service.CalendarService;
import gov.va.med.mhv.calendar.service.ToDoTaskService;
import gov.va.med.mhv.calendar.service.UserCalViewService;
import gov.va.med.mhv.calendar.util.CommonUtility;
import gov.va.med.mhv.calendar.util.MHVEventComparator;
import gov.va.med.mhv.calendar.util.MHVEventTypeComparator;
import gov.va.med.mhv.common.api.cache.CacheHandler;
import gov.va.med.mhv.common.api.exception.MHVException;
import gov.va.med.mhv.common.api.transfer.Session;
import gov.va.med.mhv.getcare.common.dto.AppointmentDTO;
import gov.va.med.mhv.getcare.service.AppointmentService;
import gov.va.med.mhv.mrp.common.dto.MyGoalTaskSearchDTO;
import gov.va.med.mhv.mrp.common.dto.MyTaskDTO;
import gov.va.med.mhv.mrp.webservice.MyGoalsWebService;
import gov.va.med.mhv.usermgmt.service.AccountActivityLogService;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.List;

import javax.annotation.Resource;
import javax.ws.rs.core.Response;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;

public class CalendarServiceImpl implements CalendarService{
	
	private static Logger log = LogManager.getLogger(CalendarServiceImpl.class);
	private static final String PERSONAL_EVT ="Personal Events";
	private static final String TO_DO_EVT ="Todo";
	private Date evtStartDate;
	private Date evtEndDate;
	
	@Resource(name = "appointmentServiceProxy")
	private AppointmentService appointmentService;
	
	@Resource(name = "myGoalsServiceProxy")
	private MyGoalsWebService myGoalsService;
	
	@Resource
	private AccountActivityLogService activityServiceProxy;

	
	@Autowired
	private CalendarEventService calendarEventService;
	
	@Autowired
	private ToDoTaskService  toDoTaskService;
	
	@Autowired
	private UserCalViewService userCalViewService;

	@Override
	public List<MHVEventModel> findEventsForUser(CalendarSearchCriteriaDTO searchCriteria) throws MHVException {
		List<MHVEventModel> events = new ArrayList<MHVEventModel>();
		List<String> eventTypes = searchCriteria.getEventTypes();
		 if(searchCriteria.getKeyword()==null &&( eventTypes == null || eventTypes.size() == 0)){
			 return events;
		 }
		 if(searchCriteria.getRangeStart() == null || searchCriteria.getRangeEnd() == null ){
			populateDefaultRanges(searchCriteria);
		 }
		 calculateDateRanges(searchCriteria);
		 try{
			 for(String eventType : eventTypes){
				 if(eventType == null || eventType.trim().length() == 0){
					 break;
				 }
				 if(eventType.equals("Personal Events")){
					 getPersonalEvents(events,searchCriteria,eventType);
				 }else  if(eventType.equals("VA Appointments")){
					 getVAAppointments(events,searchCriteria,eventType);
				 }else if(eventType.equals("My Tasks")){
					 getMyTasks(events,searchCriteria,eventType);
				 }else if(eventType.equals("My Recovery Plan Events")){
				 }else{
					 getGenericEvents(events,searchCriteria,eventType);  
				 }
			 }
			 if(searchCriteria.getKeyword() != null){
				 getToDos(events,searchCriteria);
				 
			 }
	
			 if(events != null){
				 Collections.sort(events, new MHVEventComparator());
			 }

			 
		 }catch(MHVException e){
			 log.error(e);
			 throw e;
		 }catch(Exception e){
			 log.error(e);
			 throw new MHVException(e);
		 }
		return events;
	}
	
	@Override
	public List<MHVEventModel> searchEventsForUser(CalendarSearchCriteriaDTO searchCriteria) throws MHVException {
		List<MHVEventModel> events = new ArrayList<MHVEventModel>();
		List<String> eventTypes = searchCriteria.getEventTypes();
		 if(searchCriteria.getKeyword()==null &&( eventTypes == null || eventTypes.size() == 0)){
			 return events;
		 }
		 if(searchCriteria.getRangeStart() == null || searchCriteria.getRangeEnd() == null ){
			populateDefaultRanges(searchCriteria);
		 }
		 calculateDateRanges(searchCriteria);
		 
		 
		 try{
			 	 getPersonalEvents(events,searchCriteria,"Personal Events");
			 	 
			 	 List<CalendarCategoryDTO>  activeCategories =  getAllActiveCategories(searchCriteria.getUserProfileId());
			 	 
			 	for (CalendarCategoryDTO eventType : activeCategories) 
			 	{
			 	 
			 		
			 	 // cal getGenric Events with the  event types NOT IN  my task, va appointments , MRP Task
			 	 
			 	 getGenericEvents(events,searchCriteria,eventType.getCategoryName());
			 	 
			 	}
			 	
			 	 
			 	 
				 
				 //  
				
			 
			 if(searchCriteria.getKeyword() != null){
				 getToDos(events,searchCriteria);
				 
			 }
	
			 if(events != null){
				 Collections.sort(events, new MHVEventComparator());
			 }

			 
		 }catch(MHVException e){
			 log.error(e);
			 throw e;
		 }catch(Exception e){
			 log.error(e);
			 throw new MHVException(e);
		 }
		return events;
	}



	private void calculateDateRanges(CalendarSearchCriteriaDTO searchCriteria) {
		 String strStrDate=CommonUtility.dateToString(searchCriteria.getRangeStart(), "MM/dd/yyyy");
		 String strEndDate=CommonUtility.dateToString(searchCriteria.getRangeEnd(), "MM/dd/yyyy");
		 Date stDate=CommonUtility.stringToDate(strStrDate,"MM/dd/yyyy");
		 Date endDate=CommonUtility.stringToDate(strEndDate,"MM/dd/yyyy");
		 evtStartDate=CommonUtility.getDaTeTime(stDate,"0","0");
		 evtEndDate=CommonUtility.getDaTeTime(endDate,"23","59");
	}


	private void getToDos(List<MHVEventModel> events,CalendarSearchCriteriaDTO searchCriteria) throws MHVException {
		List<ToDoTaskDTO> todos=findTodos(searchCriteria);
		if(todos != null){
			List<MHVEventModel> allEvents=loadTodos(todos,TO_DO_EVT);
			for( MHVEventModel mhvevent:allEvents){
				events.add(mhvevent);
			}
		}
	}


	private List<MHVEventModel> loadTodos(List<ToDoTaskDTO> todos,String toDoEvt) {
		List<MHVEventModel> todoList = new ArrayList<MHVEventModel>();
		for(ToDoTaskDTO todo: todos){
			MHVEventModel event = new MHVEventModel();
			event.setStartDate(todo.getDueDate());
			event.setEndDate(todo.getDueDate());
			event.setDescription(todo.getDescription());
			event.setStatus(todo.getDisplayStatus());
			event.setEventType(toDoEvt);
			event.setEventName(todo.getSubject());
			event.setEventId(todo.getToDoId());
			todoList.add(event);
		}
		return todoList;
	}


	private List<ToDoTaskDTO> findTodos(CalendarSearchCriteriaDTO searchCriteria) throws MHVException {
		List<ToDoTaskDTO> todos=null;
		try {
			todos =toDoTaskService.searchToDosForUser(searchCriteria.getUserProfileId(), searchCriteria.getKeyword());
		} catch (MHVException e) {
			throw e;
		}catch (Exception e) {
        	log.error("Error in Finding Todos", e);
        	throw new MHVException(e);
		}
		return todos;
	}


	private void getMyTasks(List<MHVEventModel> events, CalendarSearchCriteriaDTO searchCriteria,String evtType) throws MHVException {
		List<MyTaskDTO> myTasks=findMyTasks(searchCriteria);
		if(myTasks != null){
			List<MHVEventModel> allEvents=loadTaskEvents(myTasks,evtType,searchCriteria);
			for( MHVEventModel mhvevent:allEvents){
				if(searchCriteria.getKeyword() != null){
					events.add(mhvevent);
				}else{
					if(mhvevent.getStartDate().after(evtEndDate)||
					   mhvevent.getStartDate().before(evtStartDate)){
					}else{
					  events.add(mhvevent);
					}
			    }
			}
		}
		
	}
	
	
	private List<MyTaskDTO> findMyTasks(CalendarSearchCriteriaDTO searchCriteria) throws MHVException {
		List<MyTaskDTO> taskList=null;
		try {
			MyGoalTaskSearchDTO searchDto = new MyGoalTaskSearchDTO();
			searchDto.setRangeStart(searchCriteria.getRangeStart());
			searchDto.setRangeEnd(searchCriteria.getRangeEnd());
			searchDto.setUserProfileId(searchCriteria.getUserProfileId());
			searchDto.setKeyword(searchCriteria.getKeyword());
			if(searchCriteria.getKeyword() != null && searchCriteria.getKeyword().trim().length() > 0){
		    	taskList = myGoalsService.getMyGoalTasksByKeyword(searchDto);
			}else{
				taskList = myGoalsService.getMyGoalTasksForUser(searchDto);
			}
		} catch(Exception e){
			log.error("Error in Find My Task Events", e);
        	throw new MHVException(e);
		}
		return taskList;
	}
	
	
	private List<MHVEventModel> loadTaskEvents(List<MyTaskDTO> tasks,String evtType, CalendarSearchCriteriaDTO searchCriteria) {
		
		List<MHVEventModel> events = new ArrayList<MHVEventModel>();
		
		for(MyTaskDTO mytask : tasks){
			MHVEventModel event = new MHVEventModel();
			
			if(mytask.getStartTime() != null){
			   event.setStartDate(mytask.getStartTime().getTime());
			}else{
				event.setStartDate(mytask.getStartDate());
			}
			if(mytask.getEndTime() != null){
			   event.setEndDate(mytask.getEndTime().getTime());
			}else{
				if(mytask.getEndDate() != null){
					event.setEndDate(mytask.getEndDate());
				}else{
					event.setEndDate(event.getStartDate());
				}
			}
			
			if(	event.getStartDate().equals(event.getEndDate()) && (mytask.getEndTime()== null || CommonUtility.dateToString(event.getEndDate(), "hh:mma").equalsIgnoreCase("12:00AM"))){
					event.setAllDay(true);
			}
			/*
			if(CommonUtility.dateToString(event.getStartDate(), "hh:mma").equalsIgnoreCase("12:00AM")){
				event.setAllDay(true);
			} */
			
			
			event.setBeginDate(event.getStartDate());
			event.setFinishDate(mytask.getEndDate());
			event.setLocation("[My Tasks] "+ mytask.getGoalName());
			event.setEventType(evtType);
			event.setTitle("[My Tasks] "+ mytask.getTaskName());
			event.setEventName("[My Tasks] "+ mytask.getTaskName());
			event.setDescription(mytask.getDescription());
			populateRepeatInterval(event,mytask);
			
			if(mytask.getCompletedDate() != null){
				event.setRepeatEndDate(CommonUtility.getDaTeTime(mytask.getCompletedDate().getTime(),"0","0"));
			}else if(mytask.getEndDate() != null){
				event.setRepeatEndDate(CommonUtility.getDaTeTime(mytask.getEndDate(),"0","0"));
			}else{
				event.setRepeatEndDate(searchCriteria.getRangeEnd());
			}
			
			if(isEventEligible(event.getRepeatInterval(), event.getRepeatEndDate(), event.getStartDate())){
				events.add(event);
			}
			
			if(!event.getRepeatInterval().equals("NOREPEAT")){
				List<MHVEventModel> recurEvents = getRecurringEvents(event,evtType,false);
				for(MHVEventModel recurEvent : recurEvents){
					events.add(recurEvent);
				}
			}
		}
		return events;
	}
	

	private void populateRepeatInterval(MHVEventModel event, MyTaskDTO mytask) {
		if(mytask.getTaskRepetitionCycleType() == 1){
			event.setRepeatInterval("NOREPEAT");
		}else if(mytask.getTaskRepetitionCycleType() == 3){
			event.setRepeatInterval("DAILY");
		}else if(mytask.getTaskRepetitionCycleType() == 4){
			event.setRepeatInterval("MWF");
		}else if(mytask.getTaskRepetitionCycleType() == 5){
			event.setRepeatInterval("TTH");
		}else if(mytask.getTaskRepetitionCycleType() == 6){
			event.setRepeatInterval("MTWThF");
		}else if(mytask.getTaskRepetitionCycleType() == 7){
			event.setRepeatInterval("WEEKLY");
		}else if(mytask.getTaskRepetitionCycleType() == 8){
			event.setRepeatInterval("ALTWEEK");
		}else if(mytask.getTaskRepetitionCycleType() == 9){
			event.setRepeatInterval("MONTHLY");
		}else if(mytask.getTaskRepetitionCycleType() == 10){
			event.setRepeatInterval("YEARLY");
		}
	}


	private void getVAAppointments(List<MHVEventModel> events,CalendarSearchCriteriaDTO searchCriteria,String evtType) throws MHVException {
		List<AppointmentDTO> appointments=findVAAppointments(searchCriteria);
		if(appointments != null){
			loadVAAppointments(appointments,events,evtType);
		}
	}
	
	
	private List<AppointmentDTO> findVAAppointments(CalendarSearchCriteriaDTO searchCriteria) throws MHVException {
		List<AppointmentDTO> dtoList = null;
		try {
			List<AppointmentDTO> appointmentList=appointmentService.findAppointmentsForPatient(searchCriteria.getPatientId(),false);
			if(appointmentList != null){
				dtoList=new ArrayList<AppointmentDTO>();
				String keyword = searchCriteria.getKeyword() != null ? searchCriteria.getKeyword().toUpperCase() :null;
				for(AppointmentDTO appointment:appointmentList){
					if(keyword != null){
						if(appointment.getLocation().toUpperCase().contains(keyword) || 
						   appointment.getClinicName().toUpperCase().contains(keyword)){
						  dtoList.add(appointment);
						}
					}else{
						if(appointment.getApptDatePrecise() != null){
							if(appointment.getApptDatePrecise().after(evtEndDate)||
							   appointment.getApptDatePrecise().before(evtStartDate)){
							}else{
								dtoList.add(appointment);
							}
						}
				    }
				}
			}
		} catch (MHVException e) {
			throw e;
		}catch (Exception e) {
        	log.error("Error in Find VA appointments", e);
        	throw new MHVException(e);
		}
		return dtoList;
	}
	
	
	private void loadVAAppointments(List<AppointmentDTO> appointments,List<MHVEventModel> events,String evtType) {
		for(AppointmentDTO appointment: appointments){
			MHVEventModel event = new MHVEventModel();
			event.setStartDate(appointment.getApptDatePrecise());
			event.setEndDate(appointment.getApptDatePrecise());
			event.setLocation(appointment.getLocation());
			event.setClinic(appointment.getClinicName());
			event.setClinicPhone(appointment.getClinicPhone());
			event.setStatus(appointment.getDisplayStatus());
			event.setEventType(evtType);
			event.setTitle("[VA] "+appointment.getLocation()+ " Appointment");
			event.setEventName("[VA] "+appointment.getLocation()+ " Appointment");
			events.add(event);
		}
	}
	
	private void getGenericEvents(List<MHVEventModel> events,CalendarSearchCriteriaDTO searchCriteria, String eventType) throws MHVException {
		List<CalendarEventDTO> genericEvents=findGenericEvents(searchCriteria,eventType);
		if(genericEvents != null){
			List<MHVEventModel> allEvents=loadEvents(genericEvents,eventType,searchCriteria.getViewType());
			for( MHVEventModel mhvevent:allEvents){
				if(searchCriteria.getKeyword() != null){
					events.add(mhvevent);
				}else{
					if(mhvevent.getStartDate().after(evtEndDate)||
					   mhvevent.getStartDate().before(evtStartDate)){
					}else{
					  events.add(mhvevent);
					}
			    }
			}
		}
	}
	
	
	private List<CalendarEventDTO> findGenericEvents(CalendarSearchCriteriaDTO searchCriteria, String eventType) throws MHVException {
		List<CalendarEventDTO> eventList=null;
		try {
			searchCriteria.setSelectedGenericEvent(eventType);
			eventList = calendarEventService.getGenericEventsForUser(searchCriteria);
		} catch (MHVException e) {
			throw e;
		}catch(Exception e){
			log.error("Error in Find generic Events", e);
        	throw new MHVException(e);
		}
		return eventList;
	}


	
	private void getPersonalEvents(List<MHVEventModel> events, CalendarSearchCriteriaDTO searchCriteria,String evtType) throws MHVException {
		
		List<CalendarEventDTO> calEvents =findPersonalEvents(searchCriteria);
		
		if(calEvents != null){
			List<MHVEventModel> allEvents=loadEvents(calEvents,evtType,searchCriteria.getViewType());
			for( MHVEventModel mhvevent:allEvents){
				if(searchCriteria.getKeyword() != null){
					events.add(mhvevent);
				}else{
					if(mhvevent.getStartDate().after(evtEndDate)||
					   mhvevent.getStartDate().before(evtStartDate)){
					}else{
					  events.add(mhvevent);
					}
			    }
			}
		}
	}

	


	private List<CalendarEventDTO>  findPersonalEvents (CalendarSearchCriteriaDTO searchCriteria) throws MHVException {
		List<CalendarEventDTO> calEvents=null;
		try {
			 calEvents =calendarEventService.getPersonalEventsForUser(searchCriteria);
		} catch (MHVException e) {
			throw e;
		}catch (Exception e) {
        	log.error("Error in Find VA appointments", e);
        	throw new MHVException(e);
		}
		return calEvents;
	}
	
	private List<MHVEventModel> loadEvents(List<CalendarEventDTO> calEvents,String evtType,String viewType) {
		
		List<MHVEventModel> events = new ArrayList<MHVEventModel>();
		
		for(CalendarEventDTO calEvent : calEvents){
			MHVEventModel event = new MHVEventModel();
			event.setStartDate(calEvent.getStartDate());
			event.setEndDate(calEvent.getEndDate());
			event.setBeginDate(calEvent.getStartDate());
			event.setFinishDate(calEvent.getEndDate());
			event.setLocation(calEvent.getLocation());
			event.setEventType(evtType);
			event.setTitle(evtType.equals(PERSONAL_EVT) ?calEvent.getSubject():"[VA] "+ calEvent.getSubject() );
			event.setEventName(evtType.equals(PERSONAL_EVT) ?calEvent.getSubject():"[VA] "+ calEvent.getSubject() );
			event.setEventId(calEvent.getEventId());
			event.setDescription(calEvent.getDescription());
			event.setAllDay(calEvent.getAllDay());
			event.setRepeatInterval(calEvent.getRecurrenceInterval());
			event.setRepeatEndDate(calEvent.getRecurUntilDate());
			boolean multidayEvent=IsMultiDayEvent(calEvent);
			if(isEventEligible(calEvent.getRecurrenceInterval(),calEvent.getRecurUntilDate(),calEvent.getStartDate())){
			   events.add(event);
			}
			if((calEvent.getRecurrenceInterval() != null && calEvent.getRecurUntilDate() != null) ||(multidayEvent )){
				if(multidayEvent){
					event.setRepeatEndDate(CommonUtility.getDaTeTime(calEvent.getEndDate(),"0","0"));
					event.setRepeatInterval("DAILY");
				}
				event.setRepeatEndDate(CommonUtility.getDaTeTime(event.getRepeatEndDate(),"0","0"));
				List<MHVEventModel> recurEvents = getRecurringEvents(event,evtType,multidayEvent);
				for(MHVEventModel recurEvent : recurEvents){
					if(multidayEvent){
						String startDate=CommonUtility.dateToString(recurEvent.getStartDate(),"MM/dd/yyyy");
						String firstDate=CommonUtility.dateToString(event.getStartDate(),"MM/dd/yyyy");
						String lastDate=CommonUtility.dateToString(event.getEndDate(),"MM/dd/yyyy");
						
						if(!startDate.equals(firstDate) && !startDate.equals(lastDate)){
							recurEvent.setStartDate(CommonUtility.getDaTeTime(recurEvent.getStartDate(),"0","0"));
							recurEvent.setEndDate(CommonUtility.getDaTeTime(recurEvent.getStartDate(),"23","59"));
						}else if(startDate.equals(lastDate)){
							recurEvent.setStartDate(CommonUtility.getDaTeTime(recurEvent.getStartDate(),"0","0"));
							recurEvent.setEndDate(CommonUtility
								.getDaTeTime(recurEvent.getStartDate(),CommonUtility.dateToString(event.getEndDate(),"HH"),CommonUtility.dateToString(event.getEndDate(),"mm")));
			
						}
					}
					
					
					events.add(recurEvent);
				}
				if(multidayEvent){
					event.setRepeatEndDate(null);
					event.setRepeatInterval(null);
					event.setEndDate(CommonUtility.getDaTeTime(event.getStartDate(),"23","59"));
				}
		
			}
		}
		return events;
	}
	
	
	private boolean isEventEligible(String recurInterval,Date recurUntilDate,Date startDate) {
		boolean eligible=false;
		if((recurInterval != null && recurUntilDate != null)){
			Calendar cal=Calendar.getInstance();
			cal.setTime(CommonUtility.getDaTeTime(startDate,"0","0"));
			if(recurInterval.equals("MWF")){
				  if(cal.get(Calendar.DAY_OF_WEEK) == Calendar.MONDAY    ||
					 cal.get(Calendar.DAY_OF_WEEK) == Calendar.WEDNESDAY ||
					 cal.get(Calendar.DAY_OF_WEEK) == Calendar.FRIDAY){
					  
					 eligible=true;
				  }
			}else if(recurInterval.equals("TTH")){
					if(cal.get(Calendar.DAY_OF_WEEK) == Calendar.TUESDAY ||
					   cal.get(Calendar.DAY_OF_WEEK) == Calendar.THURSDAY) {
						
						eligible=true;
					}
			}else if(recurInterval.equals("MTWThF")){
					if(cal.get(Calendar.DAY_OF_WEEK) == Calendar.MONDAY ||
					   cal.get(Calendar.DAY_OF_WEEK) == Calendar.TUESDAY ||
					   cal.get(Calendar.DAY_OF_WEEK) == Calendar.WEDNESDAY ||
					   cal.get(Calendar.DAY_OF_WEEK) == Calendar.THURSDAY ||
					   cal.get(Calendar.DAY_OF_WEEK) == Calendar.FRIDAY){
						
					   eligible=true;
					}
			}else{
				eligible=true;
			}
		}else{
			eligible=true;
		}
		return eligible;
	}


	private boolean IsMultiDayEvent(CalendarEventDTO calEvent ){
		boolean multiDayEvt=false;
		if(calEvent.getRecurrenceInterval() == null && calEvent.getRecurUntilDate() == null){
		   multiDayEvt=CommonUtility.dateToString(calEvent.getStartDate(), "MM/dd/yyyy").equals
		              (CommonUtility.dateToString(calEvent.getEndDate(), "MM/dd/yyyy"))?false:true;
		}
		return multiDayEvt;
	}
	
	
	
	private void populateDefaultRanges(CalendarSearchCriteriaDTO searchCriteria) {
		Calendar cal = Calendar.getInstance();
		cal.set(Calendar.HOUR, 0);
		cal.set(Calendar.MINUTE,0);
		cal.set(Calendar.SECOND,0);
		cal.add(Calendar.YEAR, 1);
		searchCriteria.setRangeEnd(cal.getTime());
		cal.add(Calendar.YEAR, -2);
    	cal.set(Calendar.HOUR, 23);
    	cal.set(Calendar.MINUTE, 59);
    	cal.set(Calendar.SECOND, 59);

		searchCriteria.setRangeStart(cal.getTime());
	}


	private List<MHVEventModel> getRecurringEvents(MHVEventModel event,String eventType,boolean multiDayEvent) {
		List<MHVEventModel> recurEvents = new ArrayList<MHVEventModel>();
		Date recurEndDate = event.getRepeatEndDate();
		Calendar cal = Calendar.getInstance();
		cal.setTime(recurEndDate);
		cal.set(Calendar.HOUR, 23);
		cal.set(Calendar.MINUTE,59);
		cal.set(Calendar.SECOND,59);
		Date endDate = cal.getTime();
		Date beginDate = event.getStartDate();
		cal.setTime(beginDate);
		Date startDate=cal.getTime();
		Date tempDate = cal.getTime();
		
		Calendar calEnd = Calendar.getInstance();
		calEnd.setTime(event.getEndDate());

		

		for(;;){
			if(event.getRepeatInterval().equals("DAILY")){
				cal.setTime(tempDate);
				cal.add(Calendar.DAY_OF_YEAR, 1);
				calEnd.add(Calendar.DAY_OF_YEAR, 1);
				Date nextDate=cal.getTime();
				Date nextEndDate=calEnd.getTime();
				if(nextDate.after(endDate)){
					break;
				}else{
					MHVEventModel newevent = copyEvent(event,eventType,multiDayEvent);
					newevent.setStartDate(nextDate);
					newevent.setEndDate(nextEndDate);
					tempDate=nextDate;
					recurEvents.add(newevent);
				}
			}else if(event.getRepeatInterval().equals("MWF")){
				cal.setTime(tempDate);
				cal.add(Calendar.DAY_OF_YEAR, 1);
				calEnd.add(Calendar.DAY_OF_YEAR, 1);
				Date nextDate=cal.getTime();
				Date nextEndDate=calEnd.getTime();
				if(nextDate.after(endDate)){
					break;
				}else{
					if(cal.get(Calendar.DAY_OF_WEEK) == Calendar.MONDAY ||
					   cal.get(Calendar.DAY_OF_WEEK) == Calendar.WEDNESDAY ||
					   cal.get(Calendar.DAY_OF_WEEK) == Calendar.FRIDAY){
							MHVEventModel newevent = copyEvent(event,eventType,multiDayEvent);
							newevent.setStartDate(nextDate);
							newevent.setEndDate(nextEndDate);
							recurEvents.add(newevent);
					}
					tempDate=nextDate;
				}

			}else if(event.getRepeatInterval().equals("TTH")){
				cal.setTime(tempDate);
				cal.add(Calendar.DAY_OF_YEAR, 1);
				calEnd.add(Calendar.DAY_OF_YEAR, 1);
				Date nextDate=cal.getTime();
				Date nextEndDate=calEnd.getTime();
				if(nextDate.after(endDate)){
					break;
				}else{
					if(cal.get(Calendar.DAY_OF_WEEK) == Calendar.TUESDAY ||
					   cal.get(Calendar.DAY_OF_WEEK) == Calendar.THURSDAY) {
							MHVEventModel newevent = copyEvent(event,eventType,multiDayEvent);
							newevent.setStartDate(nextDate);
							newevent.setEndDate(nextEndDate);
							recurEvents.add(newevent);
					}
					tempDate=nextDate;
				}

			}else if(event.getRepeatInterval().equals("MTWThF")){
				cal.setTime(tempDate);
				cal.add(Calendar.DAY_OF_YEAR, 1);
				calEnd.add(Calendar.DAY_OF_YEAR, 1);
				Date nextDate=cal.getTime();
				Date nextEndDate=calEnd.getTime();
				if(nextDate.after(endDate)){
					break;
				}else{
					if(cal.get(Calendar.DAY_OF_WEEK) == Calendar.MONDAY ||
					   cal.get(Calendar.DAY_OF_WEEK) == Calendar.TUESDAY ||
					   cal.get(Calendar.DAY_OF_WEEK) == Calendar.WEDNESDAY ||
					   cal.get(Calendar.DAY_OF_WEEK) == Calendar.THURSDAY ||
					   cal.get(Calendar.DAY_OF_WEEK) == Calendar.FRIDAY){
							MHVEventModel newevent = copyEvent(event,eventType,multiDayEvent);
							newevent.setStartDate(nextDate);
							newevent.setEndDate(nextEndDate);
							recurEvents.add(newevent);
					}
					tempDate=nextDate;
				}

			}else if(event.getRepeatInterval().equals("MONTHLY")){
				cal.setTime(tempDate);
				cal.add(Calendar.MONTH, 1);
				calEnd.add(Calendar.MONTH, 1);
				Date nextDate=cal.getTime();
				Date nextEndDate=calEnd.getTime();
				if(nextDate.after(endDate)){
					break;
				}else{
					MHVEventModel newevent = copyEvent(event,eventType,multiDayEvent);
					newevent.setStartDate(nextDate);
					newevent.setEndDate(nextEndDate);
					recurEvents.add(newevent);
					tempDate=nextDate;
				}

			}else if(event.getRepeatInterval().equals("WEEKLY")){
				cal.setTime(tempDate);
				cal.add(Calendar.WEEK_OF_YEAR, 1);
				calEnd.add(Calendar.WEEK_OF_YEAR, 1);
				Date nextDate=cal.getTime();
				Date nextEndDate=calEnd.getTime();
				if(nextDate.after(endDate)){
					break;
				}else{
					MHVEventModel newevent = copyEvent(event,eventType,multiDayEvent);
					newevent.setStartDate(nextDate);
					newevent.setEndDate(nextEndDate);
					recurEvents.add(newevent);
					tempDate=nextDate;
				}

			}else if(event.getRepeatInterval().equals("ALTWEEK")){
				cal.setTime(tempDate);
				cal.add(Calendar.WEEK_OF_YEAR, 2);
				calEnd.add(Calendar.WEEK_OF_YEAR, 2);
				Date nextDate=cal.getTime();
				Date nextEndDate=calEnd.getTime();
				if(nextDate.after(endDate)){
					break;
				}else{
					MHVEventModel newevent = copyEvent(event,eventType,multiDayEvent);
					newevent.setStartDate(nextDate);
					newevent.setEndDate(nextEndDate);
					recurEvents.add(newevent);
					tempDate=nextDate;
				}
				
			}else if(event.getRepeatInterval().equals("YEARLY")){
				cal.setTime(tempDate);
				cal.add(Calendar.YEAR, 1);
				calEnd.add(Calendar.YEAR, 1);
				Date nextDate=cal.getTime();
				Date nextEndDate=calEnd.getTime();
				if(nextDate.after(endDate)){
					break;
				}else{
					MHVEventModel newevent = copyEvent(event,eventType,multiDayEvent);
					newevent.setStartDate(nextDate);
					newevent.setEndDate(nextEndDate);
					tempDate=nextDate;
					recurEvents.add(newevent);
				}
			}else{
				break;
			}
		}
		return recurEvents;
	}

	private MHVEventModel copyEvent(MHVEventModel event,String eventType,boolean multiDayEvent) {
		MHVEventModel newevent = new MHVEventModel();
		newevent.setStartDate(event.getStartDate());
		newevent.setEndDate(event.getEndDate());
		newevent.setBeginDate(event.getStartDate());
		if(!eventType.equals("My Tasks")){
			newevent.setFinishDate(event.getEndDate());
		}else{
			newevent.setFinishDate(event.getFinishDate());
		}
		
		newevent.setLocation(event.getLocation());
		newevent.setEventType(eventType);
		newevent.setTitle(event.getTitle());
		newevent.setEventName(event.getEventName());
		newevent.setEventId(event.getEventId());
		newevent.setDescription(event.getDescription());
		newevent.setAllDay(event.isAllDay());
		if(!multiDayEvent){
		   newevent.setRepeatInterval(event.getRepeatInterval());
		   newevent.setRepeatEndDate(event.getRepeatEndDate());
		}
		return newevent;
	}

	@Override
	public List<ToDoTaskDTO> getToDosForUser(Long userProfileId) throws MHVException  {
		List<ToDoTaskDTO> todoTasks;
		try {
			todoTasks = toDoTaskService.getToDosForUser(userProfileId);
		} catch (MHVException e) {
			throw e;
		}
		return todoTasks;
	}

	@Override
	public void saveToDoTask(ToDoTaskDTO toDoTaskDTO) throws MHVException {
		try{
	    	toDoTaskService.saveToDoTask(toDoTaskDTO);
		} catch (MHVException e) {
			throw e;
		}
	}

	@Override
	public void deleteToDoTask(Long toDoTaskId) throws MHVException  {
		try {
			toDoTaskService.deleteToDoTask(toDoTaskId);
		} catch (MHVException e) {
			throw e;
		}
		
	}

	@Override
	public List<CategoryDTO> getAllCategories(Long userProfileId) throws MHVException {
		List<CategoryDTO> categories=null;
		try {
			categories=calendarEventService.getCheckedCategories(userProfileId);
		} catch (MHVException e) {
			throw e;
		}
		return categories;
	}
	
	@Override
	public List<CalendarCategoryDTO> getAllActiveCategories(Long userProfileId) throws MHVException {
		List<CalendarCategoryDTO> categories=null;
		try {
			categories=calendarEventService.getActiveCategories(userProfileId);
		} catch (MHVException e) {
			throw e;
		}
		return categories;
	}

	@Override
	public void saveCalendarEvent(CalendarEventDTO calendarEventDto) throws MHVException  {
		try {
			calendarEventService.saveCalendarEvent(calendarEventDto);
		} catch (MHVException e) {
			throw e;
		}
		
	}

	@Override
	public void deleteCalendarEvent(Long calendarEventId) throws MHVException {
		try {
			calendarEventService.deleteCalendarEvent(calendarEventId);
		} catch (MHVException e) {
			throw e;
		}
	}

	@Override
	public void saveUserCalCatList(List<CategoryDTO> userCalCatList, Long useProfileId) throws MHVException {
		try {
			calendarEventService.saveUserCalCatList(userCalCatList, useProfileId);
		} catch (MHVException e) {
			throw e;
		}
		
	}

	@Override
	public UserCalViewDTO getUserCalViewForUser(Long userProfileId) throws MHVException	 {
		List<UserCalViewDTO> calViewDTOS=null;
		UserCalViewDTO calViewDTO=null;
		try {
			calViewDTOS=userCalViewService.getUserCalViewForUser(userProfileId);
			if(calViewDTOS != null){
			   calViewDTO=calViewDTOS.get(0);
			}
		} catch (MHVException e) {
			throw e;
		}
		return calViewDTO;
	}

	@Override
	public void saveUserCalView(UserCalViewDTO userCalViewDto) throws MHVException {
		try {
			userCalViewService.saveUserCalView(userCalViewDto);
		} catch (MHVException e) {
			throw e;
		}
	}

	@Override
	public List<CategoryDTO> getCheckedCategories(Long userProfileId) throws MHVException {
		List<CategoryDTO> categories=null;
		try {
			categories=calendarEventService.getCheckedCategories(userProfileId);
			 if(categories != null){
				 Collections.sort(categories, new MHVEventTypeComparator());
			 }
		} catch (MHVException e) {
			throw e;
		}
		return categories;
	}


	@Override
	public List<ToDoTaskDTO> getFilteredToDosForUser(Long userProfileId, String filter) throws MHVException  {
		
		List<ToDoTaskDTO> todoTasks=null;
		try {
			todoTasks=toDoTaskService.getFilteredToDosForUser(userProfileId, filter);
		} catch (MHVException e) {
			throw e;
		}
		return todoTasks;
	}


	@Override
	public Response generateReport(Long userProfileId, String rangeStart,String rangeEnd,String patientId,String eventTypes) {
		try {
			List<MHVEventModel> mhvEvents=null;
			String[] evtTypes = eventTypes.split(",");
			List<String>  eventTypeList = new ArrayList<String>();
			for(String evtType:evtTypes){
				String s1=null;
				boolean forwardSlash=false;
				if(evtType.contains("~")){
					s1=evtType.replaceAll("~", "/");
					forwardSlash=true;
				}
				if(forwardSlash){
					eventTypeList.add(s1);
				}else{
					eventTypeList.add(evtType);
				}
			}
			Date exportStartDate = CommonUtility.stringToDate(rangeStart, "MMddyyyy");
			Date exportEndDate = CommonUtility.stringToDate(rangeEnd, "MMddyyyy");
			CalendarSearchCriteriaDTO searchCriteria= new CalendarSearchCriteriaDTO();
			searchCriteria.setRangeStart(exportStartDate);
			searchCriteria.setRangeEnd(exportEndDate);
			searchCriteria.setEventTypes(eventTypeList);
			searchCriteria.setUserProfileId(userProfileId);
			searchCriteria.setViewType("listView");
			Long patid=null;
			if(patientId != null && patientId.trim().length() > 0){
				patid=Long.valueOf(patientId);
			}
			searchCriteria.setPatientId(patid);
			mhvEvents=findEventsForUser(searchCriteria);
			String result = buildCSVData(mhvEvents);
			String fileName="personal_calendar.csv";
			return Response.ok().entity(result.getBytes()).header("Content-Type","application/vnd.ms-excel").header
					("Content-Length", result.getBytes().length).header("Content-Disposition","inline; filename="+fileName).build();
			
		} catch (MHVException e) {
			return Response.status(400).build();
		}
	}


	private String buildCSVData(List<MHVEventModel> mhvEvents) {
		StringBuffer sb = new StringBuffer();
		sb.append("Subject");
		sb.append(",");
		sb.append("Start Date");
		sb.append(",");
		sb.append("Start Time");
		sb.append(",");
		sb.append("End Date");
		sb.append(",");
		sb.append("End Time");
		sb.append(",");
		sb.append("All day event");
		sb.append(",");
		sb.append("Reminder on/off");
		sb.append(",");
		sb.append("Reminder Date");
		sb.append(",");
		sb.append("Reminder Time");
		sb.append(",");
		sb.append("Categories");
		sb.append(",");
		sb.append("Description");
		sb.append(",");
		sb.append("Location");
		sb.append("\n");
		
		for(MHVEventModel event :mhvEvents){
			sb.append(event.getEventName());
			sb.append(",");
			sb.append(CommonUtility.dateToString(event.getStartDate(), "MM/dd/yyy"));
			sb.append(",");
			sb.append(CommonUtility.dateToString(event.getStartDate(), "h:mm a"));
			sb.append(",");
			sb.append(CommonUtility.dateToString(event.getEndDate(), "MM/dd/yyy"));
			sb.append(",");
			sb.append(CommonUtility.dateToString(event.getEndDate(), "h:mm a"));
			sb.append(",");
			sb.append(event.isAllDay());
			sb.append(",");
			sb.append(" ");
			sb.append(",");
			sb.append(" ");
			sb.append(",");
			sb.append(" ");
			sb.append(",");
			sb.append(event.getEventType());
			sb.append(",");
			sb.append(event.getDescription() != null ? event.getDescription():" ");
			sb.append(",");
			sb.append(event.getLocation() != null ? event.getLocation():" ");
			sb.append("\n");
		}
		
		return sb.toString();
	}


	@Override
	public MHVEventModel getPersonalEventById(Long eventId)	throws MHVException {
		CalendarEventDTO calEvent=calendarEventService.getPersonalEventById(eventId);
		
		MHVEventModel event = new MHVEventModel();
		event.setStartDate(calEvent.getStartDate());
		event.setEndDate(calEvent.getEndDate());
		event.setBeginDate(calEvent.getStartDate());
		event.setFinishDate(calEvent.getEndDate());
		event.setLocation(calEvent.getLocation());
		event.setEventType(PERSONAL_EVT);
		event.setTitle(calEvent.getSubject());
		event.setEventName(calEvent.getSubject());
		event.setEventId(calEvent.getEventId());
		event.setDescription(calEvent.getDescription());
		event.setAllDay(calEvent.getAllDay());
		event.setRepeatInterval(calEvent.getRecurrenceInterval());
		event.setRepeatEndDate(calEvent.getRecurUntilDate());
		return event;
	}


	public Date getEvtStartDate() {
		return evtStartDate;
	}


	public void setEvtStartDate(Date evtStartDate) {
		this.evtStartDate = evtStartDate;
	}


	public Date getEvtEndDate() {
		return evtEndDate;
	}


	public void setEvtEndDate(Date evtEndDate) {
		this.evtEndDate = evtEndDate;
	}

	@Override
	public void auditViewAppointments() {
		Session session = CacheHandler.getInstance().getSession();
		Long userProfileId = session.getUserId();
		activityServiceProxy.auditViewAppointmentsFromCalendar(userProfileId);
		
	}
}
